% moderation_to_tdl.m
%
% Description:
% ----------------------------------------------------------------------
% This function adapts KUCG outputs (ray_aoas, ray_delays, ray_powers) to
% link-level simulations by performing:

% 1. Set complex channel gain of each ray delays
%
% 2. Antenna filtering (As per 3GPP TR 37.840 
%                       Composite array pattern model in Table 5.4.4.2-3);
%
% 3. Delay time quantization to fit to time resolution of simulations;
%
% 4. Channel gain calculation;
%
% thereby yielding a channel sample in tapped delay line (TDL) format.
%
% No Doppler shit is considered for the fixed WPAN usage.
% ----------------------------------------------------------------------

% Copyright (c) 2025, Kyoto University, Digital Communications Laboratory | Harada Lab.
% 
% Users shall cite either [1] or [2] regarding this work from Kyoto University when using this software.
% [1] Y. Koda and H. Harada, “User Guide for Sub-THz Link-Level Evaluations Based on Kyoto University Channel Generator,” Techrxiv.
% [2] (Japanese) 香田優介，原田博司「[依頼講演] 統計的チャネルモデルに基づくミリ波・サブテラヘルツ帯チャネルシミュレータ ～利用方法と基本伝送特性評価事例～」信学技報，2025年6月．
% 
% Users shall cite the relevant articles from Kyoto University if KUCG-generated channel samples are used. 
% 
% The use of this software for any purposes directly associated with weapons or weapons of mass destruction is prohibited.
%
% Users are permitted to make secondary use of the software. Any copyrights or intellectual property rights arising from research conducted using this software 
% shall be retained by the user.
%
% The use of this software for the development of products and the provision or sale of services is permitted.
% 
% THE SOFTWARE IS PROVIDED “AS IS”, WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, 
% INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. 
% IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER INANACTION OF CONTRACT, 
% TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE
% 
% The above copyright notice and this permission notice shall be included in all copies or substantial portions of the Software.
%
% Programmed by Yusuke Koda <koda@i.kyoto-u.ac.jp>
% 

function [delay_tap_list, h_list, total_power_gain] = moderation_to_tdl(ray_aoas, ray_delays, ray_powers, tstp, directivity, seed, fc)

% ********************** Variables ***************************************
% Input values:
% ray_aoas          : AoA list of each ray in degree (One of KUCG outputs)
% ray_delays        : Delay list of each ray in seconds (One of KUCG outputs)
% ray_powers        : Normalized power list of each ray in Watt/"Sum power in Watt" (One of KUCG outputs)
% tstp              : User-set time resolution of link-level simulation
% directivity       : Indicator to use directional antenna or not ("omni" or else)
% seed              : Random seed value to alter channel gain phase per
%                     simulation loop
% fc                : Center frequency in Hz

% Output values:
% delay_tap_list    : List of delay tap
% h_list            : List of channel gain for each delay tap
% total_power_gain  : Power gain due to antenna filtering (used for power normalization)
% **********************************************************************


% ************* 1. Set complex channel gain of each ray **********

% Set random seed value
rng(seed) 

% Define channel gains of each ray by adding uniform random phase shift
ray_ampl = sqrt(ray_powers);                                 % Amplitude of each ray
ray_h = ray_ampl .* exp(1i*2*pi*rand(1, length(ray_ampl)));  % Channel gain of each ray


% ****************** 2. Filter by Receive Antenna Pattern ****************

% As per 3GPP TR 37.840 report Composite array pattern model 
% in Table 5.4.4.2-3 (You can use arbitrary antenna filtering modules)

% Exclude omni-antenna mode
if directivity == "omni"
    ray_h_filtered = ray_h;
else
    % Perform antenna filtering
    ray_h_filtered = antenna_filtering(ray_h, ray_aoas, fc);
end

% ****************** 3. Delay Time Quantization **************************

% Quantize delay in the number of delay taps
ray_delays_in_tap = floor(ray_delays ./ tstp);

% Delay tap list in ascending order
delay_tap_list = unique(ray_delays_in_tap);

% ****************** 4. Channel Gain Calculation for each tap **************************

% Channel gain list for each delay tap
h_list = zeros(1, length(delay_tap_list));

% Total power gain due to antenna filtering (W)
total_power_gain = 0;

% Calculate channel gain by adding ray h in the same quantized delay tap
for i = 1:length(delay_tap_list) % For each tap
    
    % Pick up rays in the same tap
    ray_h_in_tap = ray_h_filtered(ray_delays_in_tap == delay_tap_list(i));
    
    % Add total power in the tap
    total_power_gain = total_power_gain + sum(abs(ray_h_in_tap).^2);

    % Calculate channel gain in the tap
    h_list(i) = sum(ray_h_in_tap);
end

%**************************************************************************