% antenna_filtering.m
%
% Description:
% ----------------------------------------------------------------------
% This function filters MPCs with an receive array antenna pattern, thereby
% simulating a directional channel.
% 
% The antenna model is as per 3GPP TR37.840 Composite array pattern model
% in Table 5.4.4.2-3)
%
% The antenna consists of 16x16 passive elements with 
%       the element seperation of wavelength / 2 
%
% NOTE: Cordinate is different from 3GPP TR 37.840! In this propgam,
% positive azimuth angles are defined as the clockwise direction relative to
% the norm vector of the antenna pannel. This is OPPOSITE from the
% cordinate system in the 3GPP TR 37.840.
% ----------------------------------------------------------------------
%
% Copyright (c) 2025, Kyoto University, Digital Communications Laboratory | Harada Lab.
% 
% Users shall cite either [1] or [2] regarding this work from Kyoto University when using this software.
% [1] Y. Koda and H. Harada, “User Guide for Sub-THz Link-Level Evaluations Based on Kyoto University Channel Generator,” Techrxiv.
% [2] (Japanese) 香田優介，原田博司「[依頼講演] 統計的チャネルモデルに基づくミリ波・サブテラヘルツ帯チャネルシミュレータ ～利用方法と基本伝送特性評価事例～」信学技報，2025年6月．
% 
% Users shall cite the relevant articles from Kyoto University if KUCG-generated channel samples are used. 
% 
% The use of this software for any purposes directly associated with weapons or weapons of mass destruction is prohibited.
%
% Users are permitted to make secondary use of the software. Any copyrights or intellectual property rights arising from research conducted using this software 
% shall be retained by the user.
%
% The use of this software for the development of products and the provision or sale of services is permitted.
% 
% THE SOFTWARE IS PROVIDED “AS IS”, WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, 
% INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. 
% IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER INANACTION OF CONTRACT, 
% TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE
% 
% The above copyright notice and this permission notice shall be included in all copies or substantial portions of the Software.
%
% Programmed by Yusuke Koda <koda@i.kyoto-u.ac.jp>


function [ray_h_filtered] = antenna_filtering(ray_h, ray_aoas, fc)

% ******************** Variables ***************************************
% Input values:
% ray_aoas          : Azimuth AoA list of each ray in degree (One of KUCG outputs)
% fc                : Center frequency in Hz
% ray_h             : Complex channel gain of each ray
%
% Output values:
% ray_h_filtered    : Channel gain of each ray after the receive antenna
%                     filtering
% **********************************************************************


%***************************** Initializations ****************************

% Paramters for element pattern calculations
phi_3db = 65;   % 3 dB beamwidth in azimuth angle (deg)
a_m = 30;       % Front-back ratio (dB)
theta_3db = 65; % 3 dB beamwidth in elevation angle (deg)   
sla_v = 30;     % Siderobe level limit (dB)
g_e_max = 8;    % Maximum directional gain of the element (dB)

% Parameter for array factor calculations
wavelength = 3 * 10^8 / fc;     % Wavelength (m)
d_v = wavelength / 2;           % Array seperation in vertical domain set to half-wavelength (m)
d_h = wavelength / 2;           % Array seperation in horizontal domain set to half-wavelength (m)
num_v = 16;                     % Number of arrays in vertical domain
num_h = 16;                     % Number of arrays in horizontal domain

theta_etilt = 0;                % Boresight angle in tilt (deg)
phi_escan = 0;                  % Boresight angle in azimuth (deg)

% Set (Only azimuth directivity is considered)
ray_aoas_zenith = 90 + zeros(1, length(ray_aoas));  % (deg)


%*********** Calculate element pattern A_E (clause 5.4.4.1 in TR) *********

% Calcuate element horizontal radiation pattern A_EH
a_eh = -min(12 .* (-ray_aoas / phi_3db).^2, a_m); % (dB)

% Calcuate element vertical radiation pattern A_EV
a_ev = -min(12 .* ((ray_aoas_zenith - 90) / theta_3db).^2, sla_v);  % (dB)

% Calcuate element radiation pattern A_E (Gain in power in dB)
a_e = g_e_max - min(- a_eh + a_ev, a_m); % (dB)

% Calculate element radiation pattern in amplitude (Amplitude gain in linear scale)
a_e_amp_lin = sqrt(10.^(a_e / 10));

%*********** Calculate array factor (clause 5.4.4.3 in TR) ****************

% Steering matrix components V
V = zeros(num_h, num_v, length(ray_aoas));

% Weight matrix components W
W = zeros(num_h, num_v, length(ray_aoas));

% Transform aoa (deg) into radian
ray_aoas_rad = deg2rad(ray_aoas);                   % azimuth aoa (rad)
ray_aoas_zenith_rad = deg2rad(ray_aoas_zenith);     % zenith aoa (rad)

% Calculate Steering and Weight matrices components V and W
for i = 1:length(ray_aoas)  % Each ray
    for m = 1:num_h         % Each horizontal element
        for n = 1:num_v     % Each vertical element
            V(m, n, i) = exp(1i * 2 * pi * ...
                ((n - 1) * d_v * cos(ray_aoas_zenith_rad(i)) / wavelength ...
                + (m - 1) * d_h * sin(ray_aoas_zenith_rad(i)) * sin(-ray_aoas_rad(i)) / wavelength));
            W(m, n, i) = (1/sqrt(num_v * num_h)) * exp(1i * 2 * pi * ...
                ((n - 1) * d_v * sin(deg2rad(theta_etilt)) / wavelength ...
                - (m - 1) * d_h * cos(deg2rad(theta_etilt)) * sin(deg2rad(-phi_escan)) / wavelength));
        end
    end
end

%********************** Calculate filtered gain  **************************

ray_h_filtered = zeros(1, length(ray_aoas));

for i = 1:length(ray_aoas)  % Each ray
    ray_h_filtered(i) = a_e_amp_lin(i) * (sum(V(:,:,i) .* W(:,:,i), "all")) * ray_h(i);
end

%**************************************************************************
